#!/usr/bin/env python3
import pyhf
import json

# Load the workspace
with open("bkgonly_workspace.json") as f:
    spec = json.load(f)

# categorise regions as SR or CR
isSR = []
notSR = [] 
for channel in spec['channels']:
  if 'SR' in channel["name"] :
    isSR.append(channel["name"])
  else:
    notSR.append(channel["name"])


# now construct the workspaces
workspace= pyhf.Workspace(spec) # nominal WS
workspace_SR = workspace.prune( channels=notSR) # ONLY the SR
workspace = workspace.prune( channels=isSR) # everything except the SR, so CR-only

# start by doing to fit in the CRs only 
model = workspace.model( poi_name= "kttbar") # change the POI since mu_SIG doesn't exist in the CR-only ws
data = workspace.data(model)

# get initial param values
init_pars = model.config.suggested_init()
par_bounds = model.config.suggested_bounds()
fixed_pars = [False] * model.config.npars

# Perform background-only fit with CR-only ws
fit_result = pyhf.infer.mle.fit(
  data,
  model,
  init_pars=init_pars,
  par_bounds=par_bounds,
  fixed_params=fixed_pars,
)

# Get post-fit expected yields in CRs from CR-only fit
prefit_yields = model.expected_data(model.config.suggested_init(), include_auxdata=True)
postfit_yields = model.expected_data(fit_result, include_auxdata=True)

# Print yields per channel and bin
channels = model.config.channels
nbins = model.config.channel_nbins

offset = 0
for ch, nbin in zip(channels, nbins):
    nbin =nbins[ch]
    print(f"\nChannel: {ch}")
    for i in range(nbin):
        print(f"    Observed data: {data[offset + i]}")
        print(f"    Pre-fit background: {prefit_yields[offset + i]:.3f}")
        print(f"    Post-fit background: {postfit_yields[offset + i]:.3f}")
    offset += nbin

# now we want to propagate the fit results to the SR using the SR-only WS
model_SR = workspace_SR.model() # sr only ws
channels = model_SR.config.channels
nbins = model_SR.config.channel_nbins
init_pars_SR = model_SR.config.suggested_init()
fit_result_SR = []

# create dummy fit result with the NPs from the CR-only workspace set to their post-fit values from the CFR-only fit
for par in model_SR.config.par_order:
  if par== "mu_SIG":
    fit_result_SR.append(0) # mu_SIG needs to be set to zero (it's 1 by default, so otherwise we will see the effect of signal in the SR)
  else:
    try: # if the moodifier exists in both the SR-only and CR-only ws, retrieve its values
      par_index = model.config.par_order.index(par) 
      fit_result_SR.append(fit_result[par_index])
    except: # if the modifier exists only in the SR-only ws, set to its default value (1 or 0 depending on the type of NP)
      fit_result_SR.append(init_pars_SR[model_SR.config.par_order.index(par)])
 
# now extract the post-CR-only-fit yields in the SRs using the extrapolated NP values
prefit_yields = model_SR.expected_data(model_SR.config.suggested_init(), include_auxdata=True)
postfit_yields = model_SR.expected_data(fit_result_SR, include_auxdata=True)

# Print yields per channel and bin
channels = model_SR.config.channels
nbins = model_SR.config.channel_nbins
data = workspace_SR.data(model_SR)

offset = 0
for ch, nbin in zip(channels, nbins):
    nbin =nbins[ch]
    print(f"\nChannel: {ch}")
    for i in range(nbin):
        print(f"    Observed data: {data[offset + i]}")
        print(f"    Pre-fit background: {prefit_yields[offset + i]:.3f}")
        print(f"    Post-fit background: {postfit_yields[offset + i]:.3f}")
    offset += nbin


# Print nuisance parameters before and after the fit
print("Nuisance parameters before and after the fit:")
#for name, init_val, fit_val in zip(model.config.parameters, init_pars, fit_result):
#    print(f"{name}: initial = {init_val:.3f}, post-fit = {fit_val:.3f}")
for name in model.config.parameters:
    index = model.config.par_order.index(name) 
    init_val = model.config.suggested_init()[index]
    fit_val = fit_result[index]
    print(f"{name}: initial = {init_val:.3f}, post-fit = {fit_val:.3f}")
